<?php
/* --------------------------------------------------------------
   BaseApiV3Action.php 2021-11-11
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\Application;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\FetchMode;
use Gambio\Core\Application\Http\Request;
use Gambio\Core\Application\Http\Response;

/**
 * Class BaseApiV3Action
 *
 * @package Gambio\Api\Application
 */
class BaseApiV3Action
{
    /**
     * @var Connection
     */
    private $connection;
    
    
    /**
     * @param Connection $connection
     */
    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }
    
    
    /**
     * @param Request  $request
     * @param Response $response
     * @param array    $args
     *
     * @return Response
     */
    public function __invoke(Request $request, Response $response, array $args): Response
    {
        $uri        = $request->getUri();
        $apiBaseUrl = $uri->getScheme() . '://' . $uri->getHost() . $uri->getPath();
        
        return $response->withJson([
                                       'image-lists'       => $apiBaseUrl . '/image-lists',
                                       'options'           => $apiBaseUrl . '/options',
                                       'parcel-services'   => $apiBaseUrl . '/parcel-services',
                                       'products-options'  => $apiBaseUrl . sprintf('/products/%s/options',
                                                                                    $this->firstAdditionalOptionProductId()),
                                       'products-variants' => $apiBaseUrl . sprintf('/products/%s/variants',
                                                                                    $this->firstProductVariantsProductId()),
                                       'tracking-codes'    => $apiBaseUrl . '/tracking-codes',
                                       'withdrawals'       => $apiBaseUrl . '/withdrawals',
                                   ]);
    }
    
    
    /**
     * @return int
     */
    private function firstProductVariantsProductId(): int
    {
        $result = $this->connection->createQueryBuilder()
            ->select('products_id')
            ->from('products_properties_combis')
            ->orderBy('products_id', 'ASC')
            ->setMaxResults(1)
            ->execute()
            ->fetch(FetchMode::NUMERIC);
        
        return $result !== false ? (int)$result[0] : $this->firstProductId();
    }
    
    
    /**
     * @return int
     */
    private function firstAdditionalOptionProductId(): int
    {
        $result = $this->connection->createQueryBuilder()
            ->select('products_id')
            ->from('products_attributes')
            ->orderBy('products_id', 'ASC')
            ->setMaxResults(1)
            ->execute()
            ->fetch(FetchMode::NUMERIC);
        
        return $result !== false ? (int)$result[0] : $this->firstProductId();
    }
    
    
    /**
     * @return int
     */
    private function firstProductId(): int
    {
        $result = $this->connection->createQueryBuilder()
            ->select('products_id')
            ->from('products')
            ->orderBy('products_id', 'ASC')
            ->setMaxResults(1)
            ->execute()
            ->fetch(FetchMode::NUMERIC);
        
        return $result !== false ? (int)$result[0] : 1;
    }
}